C After db-RDA using Bray-Curtis (or other) distance, species scores 
C may be obtained by computing correlations between the species and
C the fitted site scores, also called "site scores that are linear 
C combinations of environmental variables". The correlations have to be 
C weighted as follows before being used to draw the species in biplots:
C
C                  SpeciesScore(jk) = r(jk)*s(j)/s(k)
C
C where r(jk) = correlation between species j and fitted site score vector k, 
C s(j) is the standard deviation of species j,
C and s(k) is the standard deviation of fitted site score vector k. 
C
C In standard RDA, the lengths of the species score vectors are 1 in distance 
C biplots and they are equal to the square roots of the corresponding canonical 
C eigenvalues in correlation biplots (Legendre and Legendre 1998), just as in 
C PCA. Different programs may scale the site scores in different ways, however.
C This may result in species scores that are larger than the original ones by 
C a constant, which does not change the interpretation of biplots. 
C
C This program offers the option of scaling the species scores as in standard 
C RDA, or as in program Canoco versions 3 or 4. Compared to standard RDA, 
C described in Legendre & Legendre (1998), the standard deviations of the
C site scores differ by a constant in Canoco, so that the site scores also
C differ by the following constants:
C
C    n / dsqrt(totinert)              in Canoco 3.10
C    dsqrt(n*p) / dsqrt(totinert)     in Canoco 4
C
C where n is the sumber of objects (e.g., sites), p is the number of variables 
C (species), and  'totinert' is the total inertia in the species matrix.
C
C Reference: 
C
C Legendre. P. and E. Gallagher. Ecologically meaningful transformations
C    for ordination biplots of species data. Ecology (submitted).
C
C                                             Pierre Legendre, June 1999
C ----------------------------------------------------------------------
C2345678901234567890123456789012345678901234567890123456789012345678901234567890
      Integer p,pp,pmax,ppmax,biplot
      Parameter (nmax=500,pmax=500,ppmax=100)
      Real*8 Mat1(nmax,pmax),Mat2(nmax,ppmax),Corr(pmax,ppmax),sx,sx2,
     +       ssq,xbar1(pmax),xbar2(ppmax),stdev1(pmax),stdev2(ppmax)
      Real*8 dfln,dflnm1,totinert,temp
      Character*79 namea
C
    2 write(*,*) 'Name of species data file?'
      read(*,*) namea
      open(7,file=namea,status='OLD')
      write(*,*) 'How many rows (sites) and columns (species)?'
      read(*,*) n,p
      if(n.gt.nmax) write(*,*)'Too many sites. Recompile the program.'
      if(p.gt.pmax) write(*,*)'Too many species. Recompile the program.'
      do 3 i=1,n
    3 read(7,*) (Mat1(i,j), j=1,p)
C
      write(*,*) 'Name of the "fitted site scores" file?'
      read(*,*) namea
      open(8,file=namea,status='OLD')
      write(*,*) 
     +  'How many rows (sites) and columns (fitted site score columns)?'
      read(*,*) nn,pp
      if(pp.gt.ppmax) 
     +   write(*,*)'Too many columns. Recompile the program.'
      if(nn.ne.n) then
         write(*,*) 'Error in declared number of sites.'
         write(*,*) 'Let''s start again...'
         goto 2
         endif
      do 4 i=1,n
    4 read(8,*) (Mat2(i,j), j=1,pp)
C
    5 write(*,*)
      write(*,*) 'The fitted site scores are from...'
      write(*,*) '(1) Standard RDA (Legendre & Legendre 1998, Ch. 11)'
      write(*,*) '(2) Canoco 3.1, scaling = -1 or -2'
      write(*,*) '(3) Canoco 4 (no post-transformation)'
      read(*,*) iprog
      if((iprog.lt.1).or.(iprog.gt.3)) goto 5
C
      write(*,*) 'Name of output file?'
      read(*,*) namea
      open (9,file=namea,status='NEW')
C
    6 write(*,*)
      write(*,*) 'Writing species scores onto output file: '
      write(*,*) '(1) a single line per species, or'
      write(*,*) '(2) 8 species scores per line?'
      read(*,*) ichoix
      if((ichoix.lt.1).or.(ichoix.gt.2)) goto 6
C
      dfln=dfloat(n)
      dflnm1=dfloat(n-1)
      dflp=dfloat(p)
C Standardize the species vectors
      totinert=0.0
      do 20 j=1,p
      sx=0.0
      sx2=0.0
      do 12 i=1,n
      sx=sx+Mat1(i,j)
   12 sx2=sx2+Mat1(i,j)**2
      xbar1(j)=sx/dfln
      ssq=(sx2-(sx*sx/dfln))
      stdev1(j)=dsqrt(ssq/dflnm1)
      totinert=totinert+ssq
      do 14 i=1,n
      if(stdev1(j).gt.0.0) then
         Mat1(i,j)=(Mat1(i,j)-xbar1(j))/stdev1(j)
         else
         Mat1(i,j)=0.0
         endif
   14 continue
   20 continue
C Standardize the 'fitted site scores'
      do 30 j=1,pp
      sx=0.0
      sx2=0.0
      do 22 i=1,n
      sx=sx+Mat2(i,j)
   22 sx2=sx2+Mat2(i,j)**2
      xbar2(j)=sx/dfln
      ssq=(sx2-(sx*sx/dfln))
      stdev2(j)=dsqrt(ssq/dflnm1)
      do 24 i=1,n
      if(stdev2(j).gt.0.0) then
         Mat2(i,j)=(Mat2(i,j)-xbar2(j))/stdev2(j)
         else
         Mat2(i,j)=0.0
         endif
   24 continue
   30 continue
C Compute correlations
      Call Correl(Mat1,Mat2,Corr,n,p,pp,nmax,pmax,ppmax)
C Write partial results to screen
      if((p.le.20).and.(pp.le.20)) then
         write(*,*)
         write(*,*) 'Standard deviations of species variables' 
         write(*,*)
         write(*,101) (stdev1(j), j=1,p)
         write(*,*)
         write(*,*) 'Standard deviations of the site score variables' 
         write(*,*)
         write(*,101) (stdev2(j), j=1,pp)
         endif
      if(n.le.20) then
         write(*,*)
         write(*,*) 'Correlation matrix'
         write(*,*)
         do 32 i=1,p
   32    write(*,101) (Corr(i,j), j=1,pp)
         endif
C Final scaling to obtain species scores. In standard RDA,
C Stdev2 = sqrt(canonical eigenvalues) in distance biplots
C          or 1 in correlation biplots.
C Standard deviations of the site scores differ by a constant in Canoco,
C so that the site scores also differ by the following constants:
C   n/dsqrt(totinert)             in Canoco 3.10
C   dsqrt(n*p)/dsqrt(totinert)    in Canoco 4
C
      goto (38,42,46) iprog
C
   38 do 40 j=1,p
      do 40 jj=1,pp
   40 Corr(j,jj)=Corr(j,jj)*stdev1(j)/stdev2(jj)
      goto 50
C
   42 do 44 j=1,p
      do 44 jj=1,pp
   44 Corr(j,jj)=Corr(j,jj)*stdev1(j)*dfln/(stdev2(jj)*dsqrt(totinert))
      goto 50
C
   46 temp=dsqrt(dfln*dflp)
      do 48 j=1,p
      do 48 jj=1,pp
   48 Corr(j,jj)=Corr(j,jj)*stdev1(j)*temp/(stdev2(jj)*dsqrt(totinert))
C Write results to output file
   50 if(ichoix.eq.1) then
         do 52 j=1,p
   52    write(9,101) (Corr(j,jj), jj=1,pp)
         else
         do 54 j=1,p
   54    write(9,100) (Corr(j,jj), jj=1,pp)
         endif
C Close files
      close (7)
      close (8)
      close (9)
      stop
  100 format(8f10.5)
  101 format(1000f10.5)
      end
      
      Subroutine Correl(Z1,Z2,Z3,n,p,pp,nmax,pmax,ppmax)
      Integer p,pp,pmax,ppmax
      Real*8 Z1(nmax,pmax),Z2(nmax,ppmax),Z3(pmax,ppmax),temp,dflnm1
C Correlations computed from matrices of standardized column vectors
C Z3 = Z1' * Z2
      dflnm1=dfloat(n-1)
      do 8 j=1,p
      do 8 jj=1,pp
      temp=0.0
      do 6 i=1,n
    6 temp=temp+Z1(i,j)*Z2(i,jj)
    8 Z3(j,jj)=temp/dflnm1
      return
      end
